/*
 * main.c
 *
 *  Created on: 12 Mar 2024
 *      Author: nx827554
 */

/* main.c */

#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>
#include"SmartNFCType.h"
/* 12 for UID (Length + 3 offset + 8 for UID) */
#define SMARTAG2_UID_EXTRA_LENGHT 0xC
/* Eval-SmarTag2 includes the st25dv64k */
#define STSMART_NFC_MAX_SIZE NFCTAG_64K_SIZE
/* Dimension of the CC file in bytes */
#define ST25DV_CC_SIZE            8
/* Address due to extended CC file + NDEF header before payload == (4 or 8) + 24 + SMARTAG2_UID_EXTRA_LENGHT*/
#define SMARTAG2_START_ADDR_OFFSET (0x18+ST25DV_CC_SIZE + SMARTAG2_UID_EXTRA_LENGHT)

/* NFC Protocol Version and Revision */
#define SMARTAG2_RECORD_VERSION 2
#define SMARTAG2_RECORD_REVISION 1
/* Board Id and Firmware Id */
#define SMARTAG2_BOARD_ID 1
#define SMARTAG2_FIRMWARE_ID 4

/*Formulas for encoding environmental data to a binary minimised format*/
#define STTS22H_SAMPLE_TO_CODED(Value)        ((uint16_t)(((Value)*5) - ((-10)*5)))
#define LPS22DF_SAMPLE_TO_CODED(Value)       ((uint16_t)(((Value)*2) - ((260)*2)))
#define VD6283_LUX_SAMPLE_TO_CODED(Value)    ((uint32_t)((Value)*1000))
#define SHT40_TEMP_SAMPLE_TO_CODED(Value)	 ((uint16_t)(((Value)*5) - ((-40)*5)))
#define SHT40_HUM_SAMPLE_TO_CODED(Value)	 ((uint8_t)((Value)*2))

/*Formulas for decoding environmental data from a binary minimised format*/
#define STTS22H_CODED_TO_SAMPLE(Value)       ((((float)(Value))/5)-10)
#define LPS22DF_CODED_TO_SAMPLE(Value)       ((((float)(Value))/2)+260)
#define VD6283_LUX_CODED_TO_SAMPLE(Value)     (((float)(Value))/1000)
#define SHT40_TEMP_CODED_TO_SAMPLE(Value)	 ((((float)(Value))/5)-40)
#define SHT40_HUM_CODED_TO_SAMPLE(Value)	 (((float)(Value))/2)

/* Virtual Sensor Configuration */
#define SMARTAG2_VIRTUAL_SENSORS_NUM 12
#define STTS22H_VS_ID         0
#define LPS22DF_VS_ID         1
#define VD6283_LUX_VS_ID      2
#define VD6283_CCT_VS_ID      3
#define LSM6DSOX32_VS_ID      4
#define LSM6DSOX32_6D_VS_ID   5
#define LSM6DSOX32_MLC_VS_ID  6
#define LIS2DUXS12_MLC_VS_ID  7
#define LIS2DUXS12_VS_ID      8
#define LIS2DUXS12_6D_VS_ID   9
#define SHT40_TEMP_VS_ID   	  10
#define SHT40_HUM_VS_ID   	  11

typedef enum// status of NFC
{
  STNFC_VALID_CONFIG = 0,
  STNFC_NOT_VALID_CONFIG,
  STNFC_NOT_CHANGED_CONFIG,
  STNFC_ERROR_READING_CONFIG,
} SNFC_ValidConfiguration_t;

void generateBoardFilePaths(void);
void generatePageFilePaths(int pageNo);
void readConfiguration(unsigned char*buffer, SNFC_ValidConfiguration_t *ValidConf);
void readSensorsMinMax(unsigned char *buffer, SNFC_ValidConfiguration_t *ValidConf);
void readSensorReadings(unsigned char*buffer, SNFC_ValidConfiguration_t *ValidConf, int pagesInRead, uint32_t samplesInRead);
void writeToTXT(void);
void writeToCSV(void);
void binToCSV(unsigned char *buffer, int bufferSize);

/* Exported Variables -------------------------------------------------------- */
typedef struct//environmental data
{
  time_t Deltas[100000];
  time_t Times[100000];
  float Readings[100000];

  time_t MinDelta;
  time_t MinTime;
  float Min;

  time_t MaxDelta;
  time_t MaxTime;
  float Max;

  int ArrPos;

} SensorReadings;

typedef struct {// global medicine information
	char medType[50];
	char medBrand[50];
	int dosage ; // medicine dosage in mg.
	int quantity; // number of doses in package
	time_t expiry;
}staticInfo;

typedef struct// all information for a single board
{
	char configID[17];
	int printText;
	uint8_t powerOff;
	WakeTimer_t interrupt;
	memStore_t boardMem;


	SNFC_LogDefinition_t LogDefinition;
	int NoPages;

	staticInfo medInfo;// global medicine info

	SNFC_VirtualSensor_t AllVirtualSensorsArray[SMARTAG2_VIRTUAL_SENSORS_NUM];
	SNFC_VirtualSensor_t *ConfiguratedVirtualSensorsArray[SMARTAG2_VIRTUAL_SENSORS_NUM];

	SensorReadings AllReadings[SMARTAG2_VIRTUAL_SENSORS_NUM];
}BoardValues;

#define NO_BOARDS  2//Number of boards that data will be translated from (CHANGE BEFORE RUNNING PROGRAM)

char folderPath[200] = "";// path to test folder (CHANGE BEFORE RUNNING PROGRAM)
char testName[10] = "";//name of test folder (test folders must have two folders labelled inside named input and hex with all input files being deposited in input) (CHANGE BEFORE RUNNING PROGRAM)
char fileName[30] = "data";// root of all input file names followed by board number - flash page number "0" for sensor config files (CHANGE BEFORE RUNNING PROGRAM)

time_t manualStartTime[4] = {1742914860, 1742914860, 1742914860, 1742914860};// the start times of each board tests in epoch time (for manual evaluation use)(CHANGE BEFORE RUNNING PROGRAM)
int boardNo = 0;
int showGaps = 1;

BoardValues AllBoards[NO_BOARDS];// initialising all board information

char fileInPath [150];// file paths
char fileOutPath [150];
char fileTxtPath [150];
char fileHexPath [150];

int latestReadPointer;

uint32_t WakeTimerSecondsValue[4] = {1,60,3600,86400};// definition in seconds for each interrupt period

/**
* @brief  Generate paths to files that will be written once per board Output text/CSV
*/
void generateBoardFilePaths(void){
	/*File paths to be used once per board*/
	char bufFolderPath[200];
	strcpy(bufFolderPath,folderPath);
	strcat(bufFolderPath,testName);
	strcat(bufFolderPath,"\\");

	char STRBuf[20];
	strcpy(fileTxtPath,bufFolderPath);// output board configs as .txt files
	strcat(fileTxtPath,fileName);
	sprintf(STRBuf, "%d",boardNo+1);
	strcat(fileTxtPath,STRBuf);
	strcat(fileTxtPath,".txt");

	strcpy(STRBuf, "\0");
	strcpy(fileOutPath,bufFolderPath);// output environmental data as .csv files
	strcat(fileOutPath,fileName);
	sprintf(STRBuf, "%d-OUT",boardNo+1);
	strcat(fileOutPath,STRBuf);
	strcat(fileOutPath,".csv");

}

/**
* @brief  Generate paths to files that will have multiple iterations used for each board: Input/Hex
* @param int pageNo Number of current flash page being read/translated
*/
void generatePageFilePaths(int pageNo){
	/*File paths to be used repeatedly per board*/
	char bufFolderPath[200];
	strcpy(bufFolderPath,folderPath);
	strcat(bufFolderPath,testName);
	strcat(bufFolderPath,"\\");

	char STRBuf[20];
	strcpy(fileInPath,bufFolderPath);// input binary files
	strcat(fileInPath,"Input\\");
	strcat(fileInPath,fileName);
	sprintf(STRBuf, "%d-%d",boardNo+1,pageNo);
	strcat(fileInPath,STRBuf);
	strcat(fileInPath,".bin");

	strcpy(STRBuf, "\0");
	strcpy(fileHexPath,bufFolderPath);// output input data as formatted hex in csv files for easy reading (used when evaluating this program)
	strcat(fileHexPath,"Hex\\");
	strcat(fileHexPath,fileName);
	sprintf(STRBuf, "%d-%d-HEX",boardNo+1,pageNo);
	strcat(fileHexPath,STRBuf);
	strcat(fileHexPath,".csv");

}

/**
* @brief  Initialise board variables
*/
void initBoardValues(void){

	for(int i = 0; i< SMARTAG2_VIRTUAL_SENSORS_NUM; i++){
		AllBoards[boardNo].AllReadings[i].ArrPos = 0;
	}

}

/**
* @brief  Read/translate the sensor configurations for one board
* @param unsigned char *buffer buffer information from file to be read/translated
* @param SNFC_ValidConfiguration_t *ValidConf Flags status of configuration to catch errors
*/
void readConfiguration(unsigned char *buffer, SNFC_ValidConfiguration_t *ValidConf){
	uint32_t DataBuf32;// initialise data buffers
	uint8_t *DataBuf8 = (uint8_t *)&DataBuf32;
	uint16_t *DataBuf16 = (uint16_t *)&DataBuf32;

	latestReadPointer = SMARTAG2_START_ADDR_OFFSET;// skip initial buffer 0x2C

	for(int i = 0; i<4; i++){// copy next 4 bytes to buffer
		DataBuf8[i] = buffer[latestReadPointer+i];
	}
	latestReadPointer+=4;

	AllBoards[boardNo].LogDefinition.VirtualSensorsNum = DataBuf8[3];

	if(AllBoards[boardNo].LogDefinition.VirtualSensorsNum>SMARTAG2_VIRTUAL_SENSORS_NUM) {// validate virtualsensors num
		printf("\tError: VirtualSensorsNum >%d\r\n",SMARTAG2_VIRTUAL_SENSORS_NUM);
		*ValidConf=STNFC_NOT_VALID_CONFIG;
	}

	if(*ValidConf==STNFC_VALID_CONFIG) {
		for(int i = 0; i<4; i++){// copy next 4 bytes to buffer
		  DataBuf8[i] = buffer[latestReadPointer+i];
		}
		latestReadPointer+=4;

		if(manualStartTime != 0){// save startdatetime: if startdatetime not updated when system started, enter as manual starttime to ensure correct times
			AllBoards[boardNo].LogDefinition.StartDateTime = manualStartTime[boardNo];
		}else{
			AllBoards[boardNo].LogDefinition.StartDateTime=DataBuf32;
		}
	}

	if(*ValidConf==STNFC_VALID_CONFIG) {// save expiry
		for(int i = 0; i<4; i++){// copy next 4 bytes to buffer
			DataBuf8[i] = buffer[latestReadPointer+i];
		}
		latestReadPointer+=4;

		AllBoards[boardNo].medInfo.expiry= (time_t)DataBuf32;
	}
	if(*ValidConf==STNFC_VALID_CONFIG) {// read Dosage, Quantity, printText, interrupt, powerOff
		for(int i = 0; i<4; i++){// copy next 4 bytes to buffer
			DataBuf8[i] = buffer[latestReadPointer+i];
		}
		latestReadPointer+=4;

		AllBoards[boardNo].medInfo.dosage = (uint16_t)((DataBuf32)&0xFFF);
		AllBoards[boardNo].medInfo.quantity = (uint16_t)((DataBuf32>>(12))&0xFFF);
		AllBoards[boardNo].printText = (uint8_t)((DataBuf32>>(12+12))&0x1);
		AllBoards[boardNo].boardMem = (uint8_t)((DataBuf32>>(12+12+1))&0x1);
		AllBoards[boardNo].interrupt = (WakeTimer_t)(((DataBuf16[1]>>(12+12+1+1)))&0x3);		//read interrupt period
		AllBoards[boardNo].powerOff = (uint8_t)(((DataBuf16[1]>>(12+12+1+1+2))&0x3));		//read poweroff flag
	}

	if(AllBoards[boardNo].boardMem == NFC){// ensure only FLah memory is used for this program
		printf("ERROR: BoardMem if NFC\r\n");
		*ValidConf=STNFC_NOT_VALID_CONFIG;
	}

	if(*ValidConf==STNFC_VALID_CONFIG) {// read ConfigID
	  for(int j = 0; j< 16; j+=4){// save 16 bytes in 4 byte chunks
		for(int i = 0; i<4; i++){// copy next 4 bytes to buffer
			DataBuf8[i] = buffer[latestReadPointer+i];
		}
		latestReadPointer += 4;

		AllBoards[boardNo].configID[j] = DataBuf8[0];
		AllBoards[boardNo].configID[j+1] = DataBuf8[1];
		AllBoards[boardNo].configID[j+2] = DataBuf8[2];
		AllBoards[boardNo].configID[j+3] = DataBuf8[3];
	  }
	}

	char bufTime[80];
	struct tm tsTime = *gmtime(&AllBoards[boardNo].LogDefinition.StartDateTime);
	strftime(bufTime, sizeof(bufTime), "%H:%M:%S %d-%m-%Y", &tsTime);// convert start time to printable format

	tsTime = *gmtime(&AllBoards[boardNo].medInfo.expiry);// convert expiry to printable time
	strftime(bufTime, sizeof(bufTime), "%H:%M:%S %d-%m-%Y", &tsTime);

	/*Read SensorConfigs*/
	if(*ValidConf==STNFC_VALID_CONFIG) {
	  /* We need to read the Virtual Sensor Configuration */
	  int32_t SensorNum;
	  for(SensorNum=0;((SensorNum<AllBoards[boardNo].LogDefinition.VirtualSensorsNum)&(*ValidConf==STNFC_VALID_CONFIG));SensorNum++) {
		  for(int i = 0; i<4; i++){
			  DataBuf8[i] = buffer[latestReadPointer+i];
		  }
		  latestReadPointer+=4;

		  switch(DataBuf32&0xF) { // if first 2 bits of databuf = sensorid read sensorconfig
		  case LSM6DSOX32_MLC_VS_ID:
			printf("\tFound LSM6DSOX32_MLC_VS_ID:\r\n");

			AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum] = &AllBoards[boardNo].AllVirtualSensorsArray[LSM6DSOX32_MLC_VS_ID];
			AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->Enable=1;
			AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType = (SNFC_ThresholdsUsage_t)((DataBuf32>>4)&0x3);
			AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->Th1.Ui8Value = (DataBuf32>>(4+2))&0xFF;
			  if( AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType<TH_LESS) {
				  AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->Th2.Ui8Value = (DataBuf32>>(4+2+8))&0xFF;
			  }

			  /* Check Virtual Sensors Incompatibility */
			  if(AllBoards[boardNo].AllVirtualSensorsArray[LSM6DSOX32_VS_ID].Enable) {
				printf("\tWarning: Incompatibility with WakeUp\r\n\tDisable WakeUp\r\n");
				AllBoards[boardNo].AllVirtualSensorsArray[LSM6DSOX32_VS_ID].Enable=0;
			  }
			  if(AllBoards[boardNo].AllVirtualSensorsArray[LSM6DSOX32_6D_VS_ID].Enable) {
				 printf("\tWarning Incompatibility with 6D Orientation\r\n\tDisable 6D Orientation\r\n");
				 AllBoards[boardNo].AllVirtualSensorsArray[LSM6DSOX32_6D_VS_ID].Enable=0;
			  }

			break;

		  case LIS2DUXS12_MLC_VS_ID:
			printf("\tFound LIS2DUXS12_MLC_VS_ID:\r\n");

			AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum] = &AllBoards[boardNo].AllVirtualSensorsArray[LIS2DUXS12_MLC_VS_ID];
			AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->Enable=1;
			AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType = (SNFC_ThresholdsUsage_t)((DataBuf32>>4)&0x3);
			AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->Th1.Ui8Value = (DataBuf32>>(4+2))&0xFF;
			  if( AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType<TH_LESS) {
				  AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->Th2.Ui8Value = (DataBuf32>>(4+2+8))&0xFF;
			  }

			  /* Check Virtual Sensors Incompatibility */
			  if(AllBoards[boardNo].AllVirtualSensorsArray[LIS2DUXS12_VS_ID].Enable) {
				printf("\tWarning: Incompatibility with WakeUp\r\n\tDisable WakeUp\r\n");
				AllBoards[boardNo].AllVirtualSensorsArray[LIS2DUXS12_VS_ID].Enable=0;
			  }
			  if(AllBoards[boardNo].AllVirtualSensorsArray[LIS2DUXS12_6D_VS_ID].Enable) {
				 printf("\tWarning Incompatibility with 6D Orientation\r\n\tDisable 6D Orientation\r\n");
				 AllBoards[boardNo].AllVirtualSensorsArray[LIS2DUXS12_6D_VS_ID].Enable=0;
			  }

			break;

		  case STTS22H_VS_ID:
			printf("\tFound STTS22H_VS_ID:\r\n");

			AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum] = &AllBoards[boardNo].AllVirtualSensorsArray[STTS22H_VS_ID];
			AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->Enable=1;
			AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType = (SNFC_ThresholdsUsage_t)((DataBuf32>>4)&0x3);
			AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->Th1.Ui16Value = (DataBuf32>>(4+2))&0x1FF;
			  if( AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType<TH_LESS) {
				  AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->Th2.Ui16Value = (DataBuf32>>(4+2+9))&0x1FF;
			  }

			   for(int i = 0; i<4; i++){
				  DataBuf8[i] = buffer[latestReadPointer+i];
			   }
			  latestReadPointer+=4;
			  AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->SampleRate = DataBuf32;

			break;

		  case LPS22DF_VS_ID:
			printf("\tFound LPS22DF_VS_ID:\r\n");

			AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum] = &AllBoards[boardNo].AllVirtualSensorsArray[LPS22DF_VS_ID];
			AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->Enable=1;
			AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType = (SNFC_ThresholdsUsage_t)((DataBuf32>>4)&0x3);
			AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->Th1.Ui16Value = (DataBuf32>>(4+2))&0x7FF;
			  if( AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType<TH_LESS) {
				  AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->Th2.Ui16Value = (DataBuf32>>(4+2+11))&0x7FF;
			  }
			   for(int i = 0; i<4; i++){
				  DataBuf8[i] = buffer[latestReadPointer+i];
			   }
			  latestReadPointer+=4;
			  AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->SampleRate = DataBuf32;

			break;

		  case VD6283_LUX_VS_ID:
			printf("\tFound VD6283_LUX_VS_ID:\r\n");

			AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum] = &AllBoards[boardNo].AllVirtualSensorsArray[VD6283_LUX_VS_ID];
			AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->Enable=1;
			AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType = (SNFC_ThresholdsUsage_t)((DataBuf32>>4)&0x3);
			AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->Th1.Ui32Value = (DataBuf32>>(4+2))&0x1FFFFFF;

			   for(int i = 0; i<4; i++){
				  DataBuf8[i] = buffer[latestReadPointer+i];
			   }
			  latestReadPointer+=4;
			  AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->SampleRate = DataBuf32;

			break;

		  case LSM6DSOX32_VS_ID:
			printf("\tFound LSM6DSOX32_VS_ID:\r\n");

			AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum] = &AllBoards[boardNo].AllVirtualSensorsArray[LSM6DSOX32_VS_ID];
			AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->Enable=1;
			AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType = (SNFC_ThresholdsUsage_t)((DataBuf32>>4)&0x3);
			AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->Th1.Ui16Value = (DataBuf32>>(4+2))&0x7FFFF;

			  /* Check Virtual Sensors Incompatibility */
			  if(AllBoards[boardNo].AllVirtualSensorsArray[LSM6DSOX32_MLC_VS_ID].Enable) {
				 printf("\tWarning Incompatibility with MLC\r\n\tDisable MLC\r\n");
				 AllBoards[boardNo].AllVirtualSensorsArray[LSM6DSOX32_MLC_VS_ID].Enable=0;
			  }

			break;

		  case LSM6DSOX32_6D_VS_ID:
			printf("\tFound LSM6DSOX32_6D_VS_ID:\r\n");

			AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum] = &AllBoards[boardNo].AllVirtualSensorsArray[LSM6DSOX32_6D_VS_ID];
			AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->Enable=1;
			AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType = (SNFC_ThresholdsUsage_t)((DataBuf32>>4)&0x3);
			AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->Th1.Ui8Value = (DataBuf32>>(4+2))&0xFF;
			  if( AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType<TH_LESS) {
				  AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->Th2.Ui8Value = (DataBuf32>>(4+2+8))&0xFF;
			  }

			  /* Check Virtual Sensors Incompatibility */
			  if(AllBoards[boardNo].AllVirtualSensorsArray[LSM6DSOX32_MLC_VS_ID].Enable) {
				 printf("\tWarning Incompatibility with MLC\r\n\tDisable MLC\r\n");
				 AllBoards[boardNo].AllVirtualSensorsArray[LSM6DSOX32_MLC_VS_ID].Enable=0;
			  }

			break;
		  case SHT40_TEMP_VS_ID:
			printf("\tFound SHT40_TEMP_VS_ID:\r\n");

			AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum] = &AllBoards[boardNo].AllVirtualSensorsArray[SHT40_TEMP_VS_ID];
			AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->Enable=1;
			AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType = (SNFC_ThresholdsUsage_t)((DataBuf32>>4)&0x3);
			AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->Th1.Ui16Value = (DataBuf32>>(4+2))&0x3FF;
			  if( AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType<TH_LESS) {
				  AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->Th2.Ui16Value = (DataBuf32>>(4+2+10))&0x3FF;
			  }

			   for(int i = 0; i<4; i++){
				  DataBuf8[i] = buffer[latestReadPointer+i];
			   }
			  latestReadPointer+=4;
			  AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->SampleRate = DataBuf32;

			break;
		  case SHT40_HUM_VS_ID:
			printf("\tFound SHT40_HUM_VS_ID:\r\n");

			AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum] = &AllBoards[boardNo].AllVirtualSensorsArray[SHT40_HUM_VS_ID];
			AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->Enable=1;
			AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType = (SNFC_ThresholdsUsage_t)((DataBuf32>>4)&0x3);
			AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->Th1.Ui8Value = (DataBuf32>>(4+2))&0xFF;
			  if( AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType<TH_LESS) {
				  AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->Th2.Ui8Value = (DataBuf32>>(4+2+8))&0xFF;
			  }

			   for(int i = 0; i<4; i++){
				  DataBuf8[i] = buffer[latestReadPointer+i];
			   }
			  latestReadPointer+=4;
			  AllBoards[boardNo].ConfiguratedVirtualSensorsArray[SensorNum]->SampleRate = DataBuf32;

			break;

		  default:
			*ValidConf=STNFC_NOT_VALID_CONFIG;
			printf("Error: Not recognized VirtualSensorID=%d\r\n",DataBuf32&0xF);
		  }

	  }
	}

	switch(*ValidConf) {//raise any errors
	  case STNFC_NOT_VALID_CONFIG:
		  printf("Not Valid Configuration present on NFC Skip it\r\n");
		  AllBoards[boardNo].LogDefinition.StartDateTime=0;
	  break;
	  case STNFC_ERROR_READING_CONFIG:
		  printf("Error Reading Configuration present on NFC Skip it\r\nTry again to write the new configuration\r\n");
		  AllBoards[boardNo].LogDefinition.StartDateTime=0;
	  break;
	  case STNFC_VALID_CONFIG:
	  break;
	  case STNFC_NOT_CHANGED_CONFIG:
		  printf("Going on with the current Configuration\r\n");
		  AllBoards[boardNo].LogDefinition.StartDateTime=0;
	  break;
	}
}

/**
* @brief  Read/translate the minimum and maximum readings from each active sensor in one board
* @param unsigned char *buffer buffer information from file to be read/translated
* @param SNFC_ValidConfiguration_t *ValidConf Flags status of configuration to catch errors
*/
void readSensorsMinMax(unsigned char *buffer, SNFC_ValidConfiguration_t *ValidConf){
	int32_t SensorNum;
	uint32_t DataBuf32;
	uint8_t *DataBuf8 = (uint8_t *)&DataBuf32;
	int MinMaxRead[SMARTAG2_VIRTUAL_SENSORS_NUM] = {0,0,0,0,0,0,0,0,0,0};

	  printf("\tReadMaxMinValues\r\n");

	  for(SensorNum=0;SensorNum<AllBoards[boardNo].LogDefinition.VirtualSensorsNum;SensorNum++) {

			if(AllBoards[boardNo].AllVirtualSensorsArray[STTS22H_VS_ID].Enable == 1 && MinMaxRead[STTS22H_VS_ID] == 0) {// read each sensors minmax once if it is activated
				for(int i = 0; i<4; i++){
					DataBuf8[i] = buffer[latestReadPointer+i];
				}
				latestReadPointer+=4;

				AllBoards[boardNo].AllReadings[STTS22H_VS_ID].MinTime = (time_t)(AllBoards[boardNo].LogDefinition.StartDateTime+(60*(DataBuf32&0xFFFFF)));
				AllBoards[boardNo].AllReadings[STTS22H_VS_ID].MinDelta = (time_t)(60*(DataBuf32&0xFFFFF));
				AllBoards[boardNo].AllReadings[STTS22H_VS_ID].Min =  STTS22H_CODED_TO_SAMPLE(DataBuf32>>20);


				for(int i = 0; i<4; i++){
					DataBuf8[i] = buffer[latestReadPointer+i];
				}
				latestReadPointer+=4;

				AllBoards[boardNo].AllReadings[STTS22H_VS_ID].MaxTime = (time_t)(AllBoards[boardNo].LogDefinition.StartDateTime+ (60*(DataBuf32&0xFFFFF)));
				AllBoards[boardNo].AllReadings[STTS22H_VS_ID].MaxDelta = (time_t)(60*(DataBuf32&0xFFFFF));
				AllBoards[boardNo].AllReadings[STTS22H_VS_ID].Max =  STTS22H_CODED_TO_SAMPLE(DataBuf32>>20);

				MinMaxRead[STTS22H_VS_ID] = 1;

			} else if(AllBoards[boardNo].AllVirtualSensorsArray[LPS22DF_VS_ID].Enable == 1 && MinMaxRead[LPS22DF_VS_ID] == 0) {
				for(int i = 0; i<4; i++){
					DataBuf8[i] = buffer[latestReadPointer+i];
				}
				latestReadPointer+=4;

				AllBoards[boardNo].AllReadings[LPS22DF_VS_ID].MinTime = (time_t)(AllBoards[boardNo].LogDefinition.StartDateTime+ (60*(DataBuf32&0xFFFFF)));
				AllBoards[boardNo].AllReadings[LPS22DF_VS_ID].MinDelta = (time_t)(60*(DataBuf32&0xFFFFF));
				AllBoards[boardNo].AllReadings[LPS22DF_VS_ID].Min =  LPS22DF_CODED_TO_SAMPLE(DataBuf32>>20);


				for(int i = 0; i<4; i++){
					DataBuf8[i] = buffer[latestReadPointer+i];
				}
				latestReadPointer+=4;

				AllBoards[boardNo].AllReadings[LPS22DF_VS_ID].MaxTime = (time_t)(AllBoards[boardNo].LogDefinition.StartDateTime+ (60*(DataBuf32&0xFFFFF)));
				AllBoards[boardNo].AllReadings[LPS22DF_VS_ID].MaxDelta = (time_t)(60*(DataBuf32&0xFFFFF));
				AllBoards[boardNo].AllReadings[LPS22DF_VS_ID].Max =  LPS22DF_CODED_TO_SAMPLE(DataBuf32>>20);

				MinMaxRead[LPS22DF_VS_ID] = 1;

			} else if(AllBoards[boardNo].AllVirtualSensorsArray[VD6283_LUX_VS_ID].Enable == 1 && MinMaxRead[VD6283_LUX_VS_ID] == 0) {
				for(int i = 0; i<4; i++){
					DataBuf8[i] = buffer[latestReadPointer+i];
				}
				latestReadPointer+=4;

				AllBoards[boardNo].AllReadings[VD6283_LUX_VS_ID].MaxTime = (time_t)(AllBoards[boardNo].LogDefinition.StartDateTime+ (60*(DataBuf32)));
				AllBoards[boardNo].AllReadings[VD6283_LUX_VS_ID].MaxDelta = (time_t)(60*(DataBuf32));

				for(int i = 0; i<4; i++){
					DataBuf8[i] = buffer[latestReadPointer+i];
				}
				latestReadPointer+=4;
				AllBoards[boardNo].AllReadings[VD6283_LUX_VS_ID].Max =  VD6283_LUX_CODED_TO_SAMPLE(DataBuf32);
				MinMaxRead[VD6283_LUX_VS_ID] = 1;

			} else if(AllBoards[boardNo].AllVirtualSensorsArray[LSM6DSOX32_VS_ID].Enable == 1 && MinMaxRead[LSM6DSOX32_VS_ID] == 0) {

				for(int i = 0; i<4; i++){
					DataBuf8[i] = buffer[latestReadPointer+i];
				}
				latestReadPointer+=4;

				AllBoards[boardNo].AllReadings[LSM6DSOX32_VS_ID].MaxTime = (time_t)(AllBoards[boardNo].LogDefinition.StartDateTime+ (60*(DataBuf32)));
				AllBoards[boardNo].AllReadings[LSM6DSOX32_VS_ID].MaxDelta = (time_t)(60*(DataBuf32));

				for(int i = 0; i<4; i++){
					DataBuf8[i] = buffer[latestReadPointer+i];
				}
				latestReadPointer+=4;
				AllBoards[boardNo].AllReadings[LSM6DSOX32_VS_ID].Max = DataBuf32;
				MinMaxRead[LSM6DSOX32_VS_ID] = 1;

			//    } else if(ConfiguratedVirtualSensorsArray[SensorNum] == &AllVirtualSensorsArray[LSM6DSOX32_6D_VS_ID]) {
			//      //There are not Max/Min for 6D orientation
			} else if(AllBoards[boardNo].AllVirtualSensorsArray[SHT40_TEMP_VS_ID].Enable == 1 && MinMaxRead[SHT40_TEMP_VS_ID] == 0) {// read each sensors minmax once if it is activated
				for(int i = 0; i<4; i++){
					DataBuf8[i] = buffer[latestReadPointer+i];
				}
				latestReadPointer+=4;

				AllBoards[boardNo].AllReadings[SHT40_TEMP_VS_ID].MinTime = (time_t)(AllBoards[boardNo].LogDefinition.StartDateTime+(60*(DataBuf32&0xFFFFF)));
				AllBoards[boardNo].AllReadings[SHT40_TEMP_VS_ID].MinDelta = (time_t)(60*(DataBuf32&0xFFFFF));
				AllBoards[boardNo].AllReadings[SHT40_TEMP_VS_ID].Min =  SHT40_TEMP_CODED_TO_SAMPLE(DataBuf32>>20);


				for(int i = 0; i<4; i++){
					DataBuf8[i] = buffer[latestReadPointer+i];
				}
				latestReadPointer+=4;

				AllBoards[boardNo].AllReadings[SHT40_TEMP_VS_ID].MaxTime = (time_t)(AllBoards[boardNo].LogDefinition.StartDateTime+ (60*(DataBuf32&0xFFFFF)));
				AllBoards[boardNo].AllReadings[SHT40_TEMP_VS_ID].MaxDelta = (time_t)(60*(DataBuf32&0xFFFFF));
				AllBoards[boardNo].AllReadings[SHT40_TEMP_VS_ID].Max =  SHT40_TEMP_CODED_TO_SAMPLE(DataBuf32>>20);

				MinMaxRead[SHT40_TEMP_VS_ID] = 1;
			}else if(AllBoards[boardNo].AllVirtualSensorsArray[SHT40_HUM_VS_ID].Enable == 1 && MinMaxRead[SHT40_HUM_VS_ID] == 0) {// read each sensors minmax once if it is activated
				for(int i = 0; i<4; i++){
					DataBuf8[i] = buffer[latestReadPointer+i];
				}
				latestReadPointer+=4;

				AllBoards[boardNo].AllReadings[SHT40_HUM_VS_ID].MinTime = (time_t)(AllBoards[boardNo].LogDefinition.StartDateTime+(60*(DataBuf32&0xFFFFF)));
				AllBoards[boardNo].AllReadings[SHT40_HUM_VS_ID].MinDelta = (time_t)(60*(DataBuf32&0xFFFFF));
				AllBoards[boardNo].AllReadings[SHT40_HUM_VS_ID].Min =  SHT40_HUM_CODED_TO_SAMPLE(DataBuf32>>20);

				for(int i = 0; i<4; i++){
					DataBuf8[i] = buffer[latestReadPointer+i];
				}
				latestReadPointer+=4;

				AllBoards[boardNo].AllReadings[SHT40_HUM_VS_ID].MaxTime = (time_t)(AllBoards[boardNo].LogDefinition.StartDateTime+ (60*(DataBuf32&0xFFFFF)));
				AllBoards[boardNo].AllReadings[SHT40_HUM_VS_ID].MaxDelta = (time_t)(60*(DataBuf32&0xFFFFF));
				AllBoards[boardNo].AllReadings[SHT40_HUM_VS_ID].Max =  SHT40_HUM_CODED_TO_SAMPLE(DataBuf32>>20);

				MinMaxRead[SHT40_HUM_VS_ID] = 1;
			}
	  }

	if(AllBoards[boardNo].AllVirtualSensorsArray[STTS22H_VS_ID].Enable == 1){// print out minmax for all read sensors
	  char bufMin[80];
	  struct tm tsMin = *gmtime(&AllBoards[boardNo].AllReadings[STTS22H_VS_ID].MinTime);
	  strftime(bufMin, sizeof(bufMin), "%Y-%m-%d %H:%M", &tsMin);

	  char bufMax[80];
	  struct tm tsMax = *gmtime(&AllBoards[boardNo].AllReadings[STTS22H_VS_ID].MaxTime);
	  strftime(bufMax, sizeof(bufMax), "%Y-%m-%d %H:%M", &tsMax);
	}
	if(AllBoards[boardNo].AllVirtualSensorsArray[LPS22DF_VS_ID].Enable == 1){
	  char bufMin[80];
	  struct tm tsMin = *gmtime(&AllBoards[boardNo].AllReadings[LPS22DF_VS_ID].MinTime);
	  strftime(bufMin, sizeof(bufMin), "%Y-%m-%d %H:%M", &tsMin);

	  char bufMax[80];
	  struct tm tsMax = *gmtime(&AllBoards[boardNo].AllReadings[LPS22DF_VS_ID].MaxTime);
	  strftime(bufMax, sizeof(bufMax), "%Y-%m-%d %H:%M", &tsMax);
	}

	if(AllBoards[boardNo].AllVirtualSensorsArray[VD6283_LUX_VS_ID].Enable == 1){
	  char bufMax[80];
	  struct tm tsMax = *gmtime(&AllBoards[boardNo].AllReadings[VD6283_LUX_VS_ID].MaxTime);
	  strftime(bufMax, sizeof(bufMax), "%Y-%m-%d %H:%M", &tsMax);
	}

	if(AllBoards[boardNo].AllVirtualSensorsArray[LSM6DSOX32_VS_ID].Enable == 1){
	  char bufMax[80];
	  struct tm tsMax = *gmtime(&AllBoards[boardNo].AllReadings[LSM6DSOX32_VS_ID].MaxTime);
	  strftime(bufMax, sizeof(bufMax), "%Y-%m-%d %H:%M", &tsMax);
	}

	if(AllBoards[boardNo].AllVirtualSensorsArray[SHT40_TEMP_VS_ID].Enable == 1){// print out minmax for all read sensors
	  char bufMin[80];
	  struct tm tsMin = *gmtime(&AllBoards[boardNo].AllReadings[SHT40_TEMP_VS_ID].MinTime);
	  strftime(bufMin, sizeof(bufMin), "%Y-%m-%d %H:%M", &tsMin);

	  char bufMax[80];
	  struct tm tsMax = *gmtime(&AllBoards[boardNo].AllReadings[SHT40_TEMP_VS_ID].MaxTime);
	  strftime(bufMax, sizeof(bufMax), "%Y-%m-%d %H:%M", &tsMax);
	}
	if(AllBoards[boardNo].AllVirtualSensorsArray[SHT40_HUM_VS_ID].Enable == 1){// print out minmax for all read sensors
	  char bufMin[80];
	  struct tm tsMin = *gmtime(&AllBoards[boardNo].AllReadings[SHT40_HUM_VS_ID].MinTime);
	  strftime(bufMin, sizeof(bufMin), "%Y-%m-%d %H:%M", &tsMin);

	  char bufMax[80];
	  struct tm tsMax = *gmtime(&AllBoards[boardNo].AllReadings[SHT40_HUM_VS_ID].MaxTime);
	  strftime(bufMax, sizeof(bufMax), "%Y-%m-%d %H:%M", &tsMax);
	}

}

/**
* @brief  Read/decode the environmental data for one board
* @param unsigned char *buffer buffer information from file to be read/decoded
* @param SNFC_ValidConfiguration_t *ValidConf Flags status of configuration to catch errors
* @param int pagesInRead number of pages in buffer
* @param uint32_t samplesInRead number of samples in buffer
*/
void readSensorReadings(unsigned char *buffer, SNFC_ValidConfiguration_t *ValidConf, int pagesInRead, uint32_t samplesInRead){
	uint32_t DataBuf32;
	uint8_t *DataBuf8 = (uint8_t *)&DataBuf32;
	static int sampleNumber = 0;
	int readSensor;
	samplesInRead += sampleNumber;

	while(sampleNumber < samplesInRead){
		if(latestReadPointer == 0x1000 && pagesInRead >1){
			latestReadPointer += 48;
		}
		if(latestReadPointer > 8188){
			printf("\t\tERROR: string too long \r\n");
			break;
		}

		for(int i = 0; i<4; i++){// extract sensor number from reading
			DataBuf8[i] = buffer[latestReadPointer+i];
		}
		latestReadPointer += 4;
		readSensor = DataBuf32&0xf;

		if(readSensor>= 0 && readSensor<= SMARTAG2_VIRTUAL_SENSORS_NUM){// validate sensor number and extract time of reading
			AllBoards[boardNo].AllReadings[readSensor].Times[AllBoards[boardNo].AllReadings[readSensor].ArrPos] = (time_t)((DataBuf32>>4)+AllBoards[boardNo].LogDefinition.StartDateTime);
			AllBoards[boardNo].AllReadings[readSensor].Deltas[AllBoards[boardNo].AllReadings[readSensor].ArrPos] = (time_t)((DataBuf32>>4));
		}else{
			printf("\t ERROR: Sensor Number Outside of Range");
			break;
		}

		for(int i = 0; i<4; i++){// extract and decode environmental data
			DataBuf8[i] = buffer[latestReadPointer+i];
		}
		latestReadPointer += 4;

		if (readSensor == STTS22H_VS_ID){// temperature
			AllBoards[boardNo].AllReadings[readSensor].Readings[AllBoards[boardNo].AllReadings[readSensor].ArrPos] = STTS22H_CODED_TO_SAMPLE((float)DataBuf32);
		}
		else if (readSensor == LPS22DF_VS_ID){// pressure
			AllBoards[boardNo].AllReadings[readSensor].Readings[AllBoards[boardNo].AllReadings[readSensor].ArrPos] = LPS22DF_CODED_TO_SAMPLE((float)DataBuf32);
		}
		else if (readSensor == VD6283_LUX_VS_ID){// luminosity
			AllBoards[boardNo].AllReadings[readSensor].Readings[AllBoards[boardNo].AllReadings[readSensor].ArrPos] =VD6283_LUX_CODED_TO_SAMPLE((float)DataBuf32);
		}
		else if (readSensor == SHT40_TEMP_VS_ID){// SHT40 Temperature
			AllBoards[boardNo].AllReadings[readSensor].Readings[AllBoards[boardNo].AllReadings[readSensor].ArrPos] =SHT40_TEMP_CODED_TO_SAMPLE((float)DataBuf32);
		}
		else if (readSensor == SHT40_HUM_VS_ID){// SHT40 Humidity
			AllBoards[boardNo].AllReadings[readSensor].Readings[AllBoards[boardNo].AllReadings[readSensor].ArrPos] =SHT40_HUM_CODED_TO_SAMPLE((float)DataBuf32);
		}
		else {
			AllBoards[boardNo].AllReadings[readSensor].Readings[AllBoards[boardNo].AllReadings[readSensor].ArrPos] =DataBuf32;
		}

		AllBoards[boardNo].AllReadings[readSensor].ArrPos++;


		sampleNumber ++;

	}
}

/**
* @brief  write the sensor configurations/minmax data/static information to a .txt file
*/
void writeToTXT(void){
	FILE *fpt;
	fpt = fopen(fileTxtPath, "w+");

	char bufSTR[20];// initialise buffers
	char bufTime[80];
	struct tm tsTime;

	printf("\tWrite Configs/MinMax to txt\r\n");

	char outSTR[500] = "\nReadSmartag2-V3.0:\t";
	strcat(outSTR, testName);
	sprintf(bufSTR, " BoardNo: %d",boardNo+1);
	strcat(outSTR, bufSTR);
	strcat(fileInPath,bufSTR);

	fprintf(fpt, outSTR);

	strcpy(outSTR,"\n\nSystem Settings:");//System Settings Section
	fprintf(fpt, outSTR);

	strcpy(outSTR,"\n\n\tStartDateTime\t");//Print StartDateTime
	tsTime = *gmtime(&AllBoards[boardNo].LogDefinition.StartDateTime);
	strftime(bufTime, sizeof(bufTime), "%Y-%m-%d %H:%M", &tsTime);
	strcat(outSTR, bufTime);
	fprintf(fpt, outSTR);

	strcpy(outSTR,"\n\n\tinterruptPeriod\t");// Print new static information
	strcat(outSTR, wakeTimerName[AllBoards[boardNo].interrupt]);
	fprintf(fpt, outSTR);
	strcpy(outSTR,"\n\tprintText\t");
	sprintf(bufSTR,"%d",AllBoards[boardNo].printText);
	strcat(outSTR,bufSTR);
	fprintf(fpt, outSTR);
	strcpy(outSTR,"\n\tpowerOff\t");
	sprintf(bufSTR,"%d",AllBoards[boardNo].powerOff);
	strcat(outSTR,bufSTR);
	fprintf(fpt, outSTR);
	strcpy(outSTR,"\n\tboardMem\t");
	strcat(outSTR,memStoreName[AllBoards[boardNo].boardMem]);
	fprintf(fpt, outSTR);


	strcpy(outSTR,"\n\nMedicine Information:");//Medicine Info Section
	fprintf(fpt, outSTR);

	strcpy(outSTR,"\n\tdosage\t\t");// print medicine information
	sprintf(bufSTR,"%d",AllBoards[boardNo].medInfo.dosage);
	strcat(outSTR, bufSTR);
	fprintf(fpt, outSTR);
	strcpy(outSTR,"\n\tquantity\t");
	sprintf(bufSTR,"%d",AllBoards[boardNo].medInfo.quantity);
	strcat(outSTR, bufSTR);
	fprintf(fpt, outSTR);
	strcpy(outSTR,"\n\texpiryDateTime\t");
	tsTime = *gmtime(&AllBoards[boardNo].medInfo.expiry);
	strftime(bufTime, sizeof(bufTime), "%Y-%m-%d %H:%M", &tsTime);
	strcat(outSTR, bufTime);
	fprintf(fpt, outSTR);


	strcpy(outSTR,"\n\n\nSensor Settings:\t\t");// Sensor Settings Section
	strcat(outSTR, AllBoards[boardNo].configID);
	fprintf(fpt, outSTR);

	for(int SensorNum=0;SensorNum<SMARTAG2_VIRTUAL_SENSORS_NUM;SensorNum++) {// check each sensor for information
		if(AllBoards[boardNo].AllVirtualSensorsArray[SensorNum].Enable == 1){
			 if (SensorNum == LSM6DSOX32_MLC_VS_ID){///////////////////////////Tiltrec
				 strcpy(outSTR,"\n\n\tTiltRec:");//print sensorname
				 fprintf(fpt, outSTR);

				 strcpy(outSTR,"\n\t\trecCon\t\t");// reccord conditions
				 strcat(outSTR, ThresholdsUsageName[AllBoards[boardNo].AllVirtualSensorsArray[SensorNum].ThsUsageType]);
				 fprintf(fpt, outSTR);

				 strcpy(outSTR,"\n\t\tthreshold1\t");// threshold1
				 sprintf(bufSTR,"%d",AllBoards[boardNo].AllVirtualSensorsArray[SensorNum].Th1.Ui8Value);
				 strcat(outSTR, bufSTR);
				 fprintf(fpt, outSTR);
				 if( AllBoards[boardNo].AllVirtualSensorsArray[SensorNum].ThsUsageType<TH_LESS) {// if threshold 2 exists
					 strcpy(outSTR,"\n\t\tthreshold2\t");// print threshold 2
					 sprintf(bufSTR,"%d",AllBoards[boardNo].AllVirtualSensorsArray[SensorNum].Th2.Ui8Value);
					 strcat(outSTR, bufSTR);
					 fprintf(fpt, outSTR);
				 }
				 // no samplerate for accelerometer sensors

			 }else if (SensorNum == STTS22H_VS_ID){///////////////////////////Temperature
				 strcpy(outSTR,"\n\n\tTemperature:");//print sensorname
				 fprintf(fpt, outSTR);

				 strcpy(outSTR,"\n\t\trecCon\t\t");// reccord conditions
				 strcat(outSTR, ThresholdsUsageName[AllBoards[boardNo].AllVirtualSensorsArray[SensorNum].ThsUsageType]);
				 fprintf(fpt, outSTR);

				 strcpy(outSTR,"\n\t\tthreshold1\t");// threshold1
				 sprintf(bufSTR,"%.1f",STTS22H_CODED_TO_SAMPLE(AllBoards[boardNo].AllVirtualSensorsArray[SensorNum].Th1.Ui16Value));
				 strcat(outSTR, bufSTR);
				 fprintf(fpt, outSTR);
				 if( AllBoards[boardNo].AllVirtualSensorsArray[SensorNum].ThsUsageType<TH_LESS) {// if threshold 2 exists
					 strcpy(outSTR,"\n\t\tthreshold2\t");// print threshold 2
					 sprintf(bufSTR,"%.1f",STTS22H_CODED_TO_SAMPLE(AllBoards[boardNo].AllVirtualSensorsArray[SensorNum].Th2.Ui16Value));
					 strcat(outSTR, bufSTR);
					 fprintf(fpt, outSTR);
				 }

				 strcpy(outSTR,"\n\t\tsampleRate\t");// print samplerate with units after
				 sprintf(bufSTR,"%d",AllBoards[boardNo].AllVirtualSensorsArray[SensorNum].SampleRate);
				 strcat(outSTR, bufSTR);
				 strcat(outSTR, " ");
				 strcat(outSTR, wakeTimerName[AllBoards[boardNo].interrupt]);
				 fprintf(fpt, outSTR);

			 }else if (SensorNum == LPS22DF_VS_ID){///////////////////////////Pressure
				 strcpy(outSTR,"\n\n\tPressure:");//print sensorname
				 fprintf(fpt, outSTR);

				 strcpy(outSTR,"\n\t\trecCon\t\t");// reccord conditions
				 strcat(outSTR, ThresholdsUsageName[AllBoards[boardNo].AllVirtualSensorsArray[SensorNum].ThsUsageType]);
				 fprintf(fpt, outSTR);

				 strcpy(outSTR,"\n\t\tthreshold1\t");// threshold1
				 sprintf(bufSTR,"%.1f",LPS22DF_CODED_TO_SAMPLE(AllBoards[boardNo].AllVirtualSensorsArray[SensorNum].Th1.Ui16Value));
				 strcat(outSTR, bufSTR);
				 fprintf(fpt, outSTR);
				 if(AllBoards[boardNo].AllVirtualSensorsArray[SensorNum].ThsUsageType<TH_LESS) {// if threshold 2 exists
					 strcpy(outSTR,"\n\t\tthreshold2\t");// print threshold 2
					 sprintf(bufSTR,"%.1f",LPS22DF_CODED_TO_SAMPLE(AllBoards[boardNo].AllVirtualSensorsArray[SensorNum].Th2.Ui16Value));
					 strcat(outSTR, bufSTR);
					 fprintf(fpt, outSTR);
				 }

				 strcpy(outSTR,"\n\t\tsampleRate\t");// print samplerate with units after
				 sprintf(bufSTR,"%d",AllBoards[boardNo].AllVirtualSensorsArray[SensorNum].SampleRate);
				 strcat(outSTR, bufSTR);
				 strcat(outSTR, " ");
				 strcat(outSTR, wakeTimerName[AllBoards[boardNo].interrupt]);
				 fprintf(fpt, outSTR);

			 }else if (SensorNum == VD6283_LUX_VS_ID){///////////////////////////Luminosity
				 strcpy(outSTR,"\n\n\tLuminosity:");//print sensorname
				 fprintf(fpt, outSTR);

				 strcpy(outSTR,"\n\t\trecCon\t\t");// reccord conditions
				 strcat(outSTR, ThresholdsUsageName[AllBoards[boardNo].AllVirtualSensorsArray[SensorNum].ThsUsageType]);
				 fprintf(fpt, outSTR);

				 strcpy(outSTR,"\n\t\tthreshold1\t");// threshold1
				 sprintf(bufSTR,"%.3f",VD6283_LUX_CODED_TO_SAMPLE(AllBoards[boardNo].AllVirtualSensorsArray[SensorNum].Th1.Ui32Value));
				 strcat(outSTR, bufSTR);
				 fprintf(fpt, outSTR);

				 strcpy(outSTR,"\n\t\tsampleRate\t");// print samplerate with units after
				 sprintf(bufSTR,"%d",AllBoards[boardNo].AllVirtualSensorsArray[SensorNum].SampleRate);
				 strcat(outSTR, bufSTR);
				 strcat(outSTR, " ");
				 strcat(outSTR, wakeTimerName[AllBoards[boardNo].interrupt]);
				 fprintf(fpt, outSTR);

			 }else if (SensorNum == LSM6DSOX32_VS_ID){///////////////////////////IMUACC
				 strcpy(outSTR,"\n\n\tIMUACC:");//print sensorname
				 fprintf(fpt, outSTR);

				 strcpy(outSTR,"\n\t\trecCon\t\t");// reccord conditions
				 strcat(outSTR, ThresholdsUsageName[AllBoards[boardNo].AllVirtualSensorsArray[SensorNum].ThsUsageType]);
				 fprintf(fpt, outSTR);

				 strcpy(outSTR,"\n\t\tthreshold1\t");// threshold1
				 sprintf(bufSTR,"%d",AllBoards[boardNo].AllVirtualSensorsArray[SensorNum].Th1.Ui16Value);
				 strcat(outSTR, bufSTR);
				 fprintf(fpt, outSTR);

			 }else if (SensorNum == LSM6DSOX32_6D_VS_ID){///////////////////////////Orientation
				 strcpy(outSTR,"\n\n\tOrientation:");
				 fprintf(fpt, outSTR);

				 strcpy(outSTR,"\n\t\trecCon\t\t");// reccord conditions
				 strcat(outSTR, ThresholdsUsageName[AllBoards[boardNo].AllVirtualSensorsArray[SensorNum].ThsUsageType]);
				 fprintf(fpt, outSTR);

				 strcpy(outSTR,"\n\t\tthreshold1\t");// threshold1
				 sprintf(bufSTR,"%d",AllBoards[boardNo].AllVirtualSensorsArray[SensorNum].Th1.Ui8Value);
				 strcat(outSTR, bufSTR);
				 fprintf(fpt, outSTR);
				 if(AllBoards[boardNo].AllVirtualSensorsArray[SensorNum].ThsUsageType<TH_LESS) {// if threshold 2 exists
					 strcpy(outSTR,"\n\t\tthreshold2\t");// print threshold 2
					 sprintf(bufSTR,"%d",AllBoards[boardNo].AllVirtualSensorsArray[SensorNum].Th2.Ui8Value);
					 strcat(outSTR, bufSTR);
					 fprintf(fpt, outSTR);
				 }


			 }else if (SensorNum == SHT40_TEMP_VS_ID){///////////////////////////SHT40Temperature
				 strcpy(outSTR,"\n\n\tSHT40 Temperature:");//print sensorname
				 fprintf(fpt, outSTR);

				 strcpy(outSTR,"\n\t\trecCon\t\t");// reccord conditions
				 strcat(outSTR, ThresholdsUsageName[AllBoards[boardNo].AllVirtualSensorsArray[SensorNum].ThsUsageType]);
				 fprintf(fpt, outSTR);

				 strcpy(outSTR,"\n\t\tthreshold1\t");// threshold1
				 sprintf(bufSTR,"%.1f",SHT40_TEMP_CODED_TO_SAMPLE(AllBoards[boardNo].AllVirtualSensorsArray[SensorNum].Th1.Ui16Value));
				 strcat(outSTR, bufSTR);
				 fprintf(fpt, outSTR);
				 if(AllBoards[boardNo].AllVirtualSensorsArray[SensorNum].ThsUsageType<TH_LESS) {// if threshold 2 exists
					 strcpy(outSTR,"\n\t\tthreshold2\t");// print threshold 2
					 sprintf(bufSTR,"%.1f",SHT40_TEMP_CODED_TO_SAMPLE(AllBoards[boardNo].AllVirtualSensorsArray[SensorNum].Th2.Ui16Value));
					 strcat(outSTR, bufSTR);
					 fprintf(fpt, outSTR);
				 }

				 strcpy(outSTR,"\n\t\tsampleRate\t");// print samplerate with units after
				 sprintf(bufSTR,"%d",AllBoards[boardNo].AllVirtualSensorsArray[SensorNum].SampleRate);
				 strcat(outSTR, bufSTR);
				 strcat(outSTR, " ");
				 strcat(outSTR, wakeTimerName[AllBoards[boardNo].interrupt]);
				 fprintf(fpt, outSTR);

			 }else if (SensorNum == SHT40_HUM_VS_ID){///////////////////////////SHT40Humidity
				 strcpy(outSTR,"\n\n\tSHT40 Humidity:");//print sensorname
				 fprintf(fpt, outSTR);

				 strcpy(outSTR,"\n\t\trecCon\t\t");// reccord conditions
				 strcat(outSTR, ThresholdsUsageName[AllBoards[boardNo].AllVirtualSensorsArray[SensorNum].ThsUsageType]);
				 fprintf(fpt, outSTR);

				 strcpy(outSTR,"\n\t\tthreshold1\t");// threshold1
				 sprintf(bufSTR,"%.1f",SHT40_HUM_CODED_TO_SAMPLE(AllBoards[boardNo].AllVirtualSensorsArray[SensorNum].Th1.Ui8Value));
				 strcat(outSTR, bufSTR);
				 fprintf(fpt, outSTR);
				 if(AllBoards[boardNo].AllVirtualSensorsArray[SensorNum].ThsUsageType<TH_LESS) {// if threshold 2 exists
					 strcpy(outSTR,"\n\t\tthreshold2\t");// print threshold 2
					 sprintf(bufSTR,"%.1f",SHT40_HUM_CODED_TO_SAMPLE(AllBoards[boardNo].AllVirtualSensorsArray[SensorNum].Th2.Ui8Value));
					 strcat(outSTR, bufSTR);
					 fprintf(fpt, outSTR);
				 }

				 strcpy(outSTR,"\n\t\tsampleRate\t");// print samplerate with units after
				 sprintf(bufSTR,"%d",AllBoards[boardNo].AllVirtualSensorsArray[SensorNum].SampleRate);
				 strcat(outSTR, bufSTR);
				 strcat(outSTR, " ");
				 strcat(outSTR, wakeTimerName[AllBoards[boardNo].interrupt]);
				 fprintf(fpt, outSTR);

			  }else{
				  strcpy(outSTR,"\n\n\tSensor_Name_Not_Found:");
				  fprintf(fpt, outSTR);
			  }
		}
	}


	strcpy(outSTR,"\n\n\nResults:");//print results title
	fprintf(fpt, outSTR);
	strcpy(outSTR,"\n\tNumber of Pages:\t");	//print Number of Pages
	sprintf(bufSTR,"%d",AllBoards[boardNo].NoPages);
	strcat(outSTR, bufSTR);
	fprintf(fpt, outSTR);

	strcpy(outSTR,"\n\tNumber of Samples:\t");	//print number of samples
	sprintf(bufSTR,"%d",AllBoards[boardNo].LogDefinition.SampleCounter);
	strcat(outSTR, bufSTR);
	fprintf(fpt, outSTR);



	for(int SensorNum=0;SensorNum<SMARTAG2_VIRTUAL_SENSORS_NUM;SensorNum++) {// print max/min of each Sensor
		if(AllBoards[boardNo].AllVirtualSensorsArray[SensorNum].Enable == 1){
			// for each sensor check if sensorNum == sensor
				//printSensorName
				// print max
				//print Min
			 if (SensorNum == STTS22H_VS_ID){///////////////////////////Temperature
				 strcpy(outSTR,"\n\n\tTemperature:");//print sensorname
				 fprintf(fpt, outSTR);

				 strcpy(outSTR,"\n\n\tMin:\t");
				 sprintf(bufSTR,"%.1f\t%d",AllBoards[boardNo].AllReadings[SensorNum].Min,AllBoards[boardNo].AllReadings[SensorNum].MinDelta);
				 strcat(outSTR, bufSTR);
				 fprintf(fpt, outSTR);

				 strcpy(outSTR,"\n\n\tMax:\t");
				 sprintf(bufSTR,"%.1f\t%d",AllBoards[boardNo].AllReadings[SensorNum].Max,AllBoards[boardNo].AllReadings[SensorNum].MaxDelta);
				 strcat(outSTR, bufSTR);
				 fprintf(fpt, outSTR);

			 }else if (SensorNum == LPS22DF_VS_ID){///////////////////////////Pressure
				 strcpy(outSTR,"\n\n\tPressure:");//print sensorname
				 fprintf(fpt, outSTR);

				 strcpy(outSTR,"\n\n\tMin:\t");
				 sprintf(bufSTR,"%.1f\t%d",AllBoards[boardNo].AllReadings[SensorNum].Min,AllBoards[boardNo].AllReadings[SensorNum].MinDelta);
				 strcat(outSTR, bufSTR);
				 fprintf(fpt, outSTR);

				 strcpy(outSTR,"\n\n\tMax:\t");
				 sprintf(bufSTR,"%.1f\t%d",AllBoards[boardNo].AllReadings[SensorNum].Max,AllBoards[boardNo].AllReadings[SensorNum].MaxDelta);
				 strcat(outSTR, bufSTR);
				 fprintf(fpt, outSTR);

			 }else if (SensorNum == VD6283_LUX_VS_ID){///////////////////////////Luminosity
				 strcpy(outSTR,"\n\n\tLuminosity:");//print sensorname
				 fprintf(fpt, outSTR);

				 strcpy(outSTR,"\n\n\tMin:\t");
				 sprintf(bufSTR,"%.3\t%d",AllBoards[boardNo].AllReadings[SensorNum].Min,AllBoards[boardNo].AllReadings[SensorNum].MinDelta);
				 strcat(outSTR, bufSTR);
				 fprintf(fpt, outSTR);

				 strcpy(outSTR,"\n\n\tMax:\t");
				 sprintf(bufSTR,"%.3f\t%d",AllBoards[boardNo].AllReadings[SensorNum].Max,AllBoards[boardNo].AllReadings[SensorNum].MaxDelta);
				 strcat(outSTR, bufSTR);
				 fprintf(fpt, outSTR);

			 }else if (SensorNum == SHT40_TEMP_VS_ID){///////////////////////////SHT40Temperature
				 strcpy(outSTR,"\n\n\tSHT40 Temperature:");//print sensorname
				 fprintf(fpt, outSTR);

				 strcpy(outSTR,"\n\n\tMin:\t");
				 sprintf(bufSTR,"%.1f\t%d",AllBoards[boardNo].AllReadings[SensorNum].Min,AllBoards[boardNo].AllReadings[SensorNum].MinDelta);
				 strcat(outSTR, bufSTR);
				 fprintf(fpt, outSTR);

				 strcpy(outSTR,"\n\n\tMax:\t");
				 sprintf(bufSTR,"%.1f\t%d",AllBoards[boardNo].AllReadings[SensorNum].Max,AllBoards[boardNo].AllReadings[SensorNum].MaxDelta);
				 strcat(outSTR, bufSTR);
				 fprintf(fpt, outSTR);

			 }else if (SensorNum == SHT40_HUM_VS_ID){///////////////////////////SHT40Humidity
				 strcpy(outSTR,"\n\n\tSHT40 Humidity:");//print sensorname
				 fprintf(fpt, outSTR);

				 strcpy(outSTR,"\n\n\tMin:\t");
				 sprintf(bufSTR,"%.1f\t%d",AllBoards[boardNo].AllReadings[SensorNum].Min,AllBoards[boardNo].AllReadings[SensorNum].MinDelta);
				 strcat(outSTR, bufSTR);
				 fprintf(fpt, outSTR);

				 strcpy(outSTR,"\n\n\tMax:\t");
				 sprintf(bufSTR,"%.1f\t%d",AllBoards[boardNo].AllReadings[SensorNum].Max,AllBoards[boardNo].AllReadings[SensorNum].MaxDelta);
				 strcat(outSTR, bufSTR);
				 fprintf(fpt, outSTR);
			 }
		}
	}

	strcpy(outSTR,"\n\n\n\nFile In\t\t");//print other file locations
	strcat(outSTR, fileInPath);
	fprintf(fpt, outSTR);
	strcpy(outSTR,"\nReadings Out\t");
	strcat(outSTR, fileOutPath);
	fprintf(fpt, outSTR);

	fclose(fpt);
}

/**
* @brief  write the environmental data to a .csv file
*/
void writeToCSV(void){
	FILE *fpt; // open csv file
	fpt = fopen(fileOutPath, "w+");
	fprintf(fpt, "\n");

	char outSTR[500] = ",";// initialise outstring

	fprintf(fpt, "\n");
	strcpy(outSTR, "\0");
	strcat(outSTR, ",");

	if(AllBoards[boardNo].AllReadings[STTS22H_VS_ID].ArrPos >0){// print titles for sensor readings
			strcat(outSTR, "Temperature,,,,");
		}
		if(AllBoards[boardNo].AllReadings[LPS22DF_VS_ID  ].ArrPos >0){
			strcat(outSTR, "Pressure,,,,");
		}
		if(AllBoards[boardNo].AllReadings[VD6283_LUX_VS_ID ].ArrPos >0){
			strcat(outSTR, "Luminosity,,,,");
		}
		if(AllBoards[boardNo].AllReadings[LSM6DSOX32_VS_ID ].ArrPos >0){
			strcat(outSTR, "IMUACC,,,,");
		}
		if(AllBoards[boardNo].AllReadings[LSM6DSOX32_6D_VS_ID].ArrPos >0){
			strcat(outSTR, "Orientation,,,,");
		}
		if(AllBoards[boardNo].AllReadings[LSM6DSOX32_MLC_VS_ID].ArrPos >0){
			strcat(outSTR, "TiltRec,,,,");
		}
		if(AllBoards[boardNo].AllReadings[SHT40_TEMP_VS_ID].ArrPos >0){
			strcat(outSTR, "SHT40Temp,,,,");
		}
		if(AllBoards[boardNo].AllReadings[SHT40_HUM_VS_ID].ArrPos >0){
			strcat(outSTR, "SHT40Hum,,,,");
		}
		strcat(outSTR,"\n");
		fprintf(fpt, outSTR);

		strcpy(outSTR, "\0");
		strcat(outSTR, ",");
		for(int i = 0; i< SMARTAG2_VIRTUAL_SENSORS_NUM; i++){// print subtitles for sensor readings
			if(i != 3){
				if(AllBoards[boardNo].AllReadings[i].ArrPos >0){
						strcat(outSTR, "Delta,Time,Reading,,");
					}
			}
		}
		strcat(outSTR,"\n");
		fprintf(fpt, outSTR);

		int MaxArrPos = 0;
		for(int i = 0; i<SMARTAG2_VIRTUAL_SENSORS_NUM; i++){// stop printing by end of sensor with largest number of reads
			if(i != 3){
				if(AllBoards[boardNo].AllReadings[i].ArrPos > MaxArrPos){
					MaxArrPos = AllBoards[boardNo].AllReadings[i].ArrPos;
				}
			}
	}


	for(int i = 0; i<= MaxArrPos; i++){//print times and readings
		strcpy(outSTR, "\0");
		strcat(outSTR, ",");
		for(int j = 0; j<SMARTAG2_VIRTUAL_SENSORS_NUM; j++){
			if(j != 3){
				if(AllBoards[boardNo].AllReadings[j].ArrPos >i){
					char STRBuf[500];
					char timeBuf[82];
					struct tm tsReading = *gmtime(&AllBoards[boardNo].AllReadings[j].Times[i]);
					strftime(timeBuf, sizeof(timeBuf), "%Y-%m-%d %H:%M:%S", &tsReading);

					sprintf(STRBuf, "%ld,%s,%f,,", AllBoards[boardNo].AllReadings[j].Deltas[i],timeBuf,AllBoards[boardNo].AllReadings[j].Readings[i]);
					strcat(outSTR, STRBuf);
				}
				else if(AllBoards[boardNo].AllReadings[j].ArrPos >0){
					strcat(outSTR, ",,,,");
				}
			}
		}
		strcat(outSTR, "\n");
		fprintf(fpt, outSTR);
	}
	fclose(fpt);
}

/**
* @brief  write the input information in hex to a .csv file
* @param unsigned char *buffer buffer information from file to be read/decoded
* @param int bufferSize buffer size of buffer string
*/
void binToCSV(unsigned char *buffer, int bufferSize){
	FILE *fpt;
	fpt = fopen(fileHexPath, "w+");
	fprintf(fpt, "\n");
	char outSTR[300];

	strcat(outSTR, "ADDR,,");
	for(int i = 0; i<4; i++){//Count to 3 for first Line
		char iChar[3];
		sprintf(iChar, "%d", i);
		strcat(outSTR, iChar);
		strcat(outSTR, ",");
	}
	strcat(outSTR, "\n");
	fprintf(fpt,outSTR);
	strcpy(outSTR, "\0");

	int k = 0;
	for(int i = 0; i< bufferSize; i++){// print binary as hexadecimal in rows of 4
		char hexOut[3];
		sprintf(hexOut,"%x,",buffer[i]);
		strcat(outSTR,hexOut);

		if(k == 3){
			strcat(outSTR, "\n");
			fprintf(fpt,outSTR);
			strcpy(outSTR, "\0");
			char iChar[2];
			sprintf(iChar, "%d", i+1);
			strcat(outSTR, iChar);
			strcat(outSTR, ",,");
			k = 0;
		}else{
			k++;
		}

	}

	fclose(fpt); // close file
}

int main(int argc, char *argv[]){
	SNFC_ValidConfiguration_t ValidConf=STNFC_VALID_CONFIG;
	uint32_t DataBuf32;
	uint8_t *DataBuf8 = (uint8_t *)&DataBuf32;
	FILE *ptr;
	unsigned char buffer[8192];// create of NFC memory in SmarTag2
	printf("Reading FlashData from %d Boards\r\n", NO_BOARDS);

	while(boardNo< NO_BOARDS){
		printf("\r\n  Start BoardNo: %d\r\n", boardNo+1);
		initBoardValues();

		generateBoardFilePaths();

		generatePageFilePaths(0);
		ptr = fopen(fileInPath, "rb");// read page to buf and write Hex csv
		fread(buffer, sizeof(buffer), 1, ptr);
		fclose(ptr);
		binToCSV(buffer,sizeof(buffer));

		/*Read page 0*/
		latestReadPointer = SMARTAG2_START_ADDR_OFFSET;// skip initial buffer 0x2C
		readConfiguration(buffer, &ValidConf);
		readSensorsMinMax(buffer, &ValidConf);
		for(int i = 0; i<4; i++){
			DataBuf8[i] = buffer[latestReadPointer+i];
		}
		latestReadPointer += 4;
		AllBoards[boardNo].LogDefinition.SampleCounter = DataBuf32;  // read sample counter

		for(int i = 0; i<4; i++){
			DataBuf8[i] = buffer[latestReadPointer+i];
		}
		latestReadPointer += 4;
		AllBoards[boardNo].NoPages = DataBuf32; // read last sample pointer

		printf("\tNoPages: %d\r\n", AllBoards[boardNo].NoPages);
		printf("\tNoSamples: %d\r\n", AllBoards[boardNo].LogDefinition.SampleCounter);
		writeToTXT();

		int pageNo = 1;
		while(pageNo<=AllBoards[boardNo].NoPages){
			int pagesInRead = 0;
			generatePageFilePaths(pageNo);

			ptr = fopen(fileInPath, "rb");// read page to buf and write Hex csv
			fread(buffer, sizeof(buffer), 1, ptr);
			fclose(ptr);
			binToCSV(buffer,sizeof(buffer));
			latestReadPointer = SMARTAG2_START_ADDR_OFFSET;// skip initial buffer 0x2C

			for(int i = 0; i<4; i++){
				DataBuf8[i] = buffer[latestReadPointer+i];
			}
			latestReadPointer += 4;

			uint8_t FlashReturnCommand = (uint8_t)((DataBuf32)&0xFF);// read page header information
			uint8_t FlashPageNo = (uint8_t)((DataBuf32>>8)&0xFF);
			uint16_t FlashSampleNo = (uint16_t)((DataBuf32>>16)&0xFFFF);
			uint16_t samplesInRead = FlashSampleNo;
			if(FlashReturnCommand != 0xFA){//handle errors in page format
				printf("\t ERROR: Page %d Incorrectly Formatted\r\n", pageNo);
				break;
			}
			if(FlashPageNo != pageNo){
				printf("\t ERROR: PageFile %d Incorrectly Named\r\n", pageNo);
				//break;
			}
			pagesInRead++;

			for(int i = 0; i<4; i++){
				DataBuf8[i] = buffer[(0x1000 + 44)+i];
			}
			if(pageNo != AllBoards[boardNo].NoPages){
				FlashReturnCommand = (uint8_t)((DataBuf32)&0xFF);// read page header information
				FlashPageNo = (uint8_t)((DataBuf32>>8)&0xFF);
				FlashSampleNo = (uint16_t)((DataBuf32>>16)&0xFFFF);
				samplesInRead += FlashSampleNo;
				if(FlashReturnCommand != 0xFA){//handle errors in page format
					printf("\t ERROR: Page %d Incorrectly Formatted\r\n", pageNo+1);
					//break;
				}
				pagesInRead++;
				if(FlashPageNo != pageNo+1){
					printf("\t ERROR: Page2File %d Incorrectly Named\r\n", pageNo+1);
					//break;
				}
			}
			readSensorReadings(buffer, &ValidConf, pagesInRead, samplesInRead);
			pageNo+= pagesInRead;
		}
		writeToCSV();
		boardNo ++;
	}
}


